<?php
/**
 * Plugin Name: うどんMP3プレーヤー (Udon MP3 Player)
 * Plugin URI:  
 * Description: 管理画面からアップロードしたMP3データをしゃれた感じで再生できるプレーヤーを記事に組み込むプラグイン。
 * Version:     1.0.001
 * Author:      Takepon
 * Author URI:  
 * Text Domain: udon-mp3-player
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

define( 'UDON_MP3_PLAYER_VERSION', '1.0.001' );
define( 'UDON_MP3_PLAYER_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'UDON_MP3_PLAYER_PLUGIN_URL', plugin_dir_url( __FILE__ ) );

/**
 * The main plugin class.
 */
class Udon_MP3_Player {

	/**
	 * Instance of this class.
	 */
	private static $instance = null;

	/**
	 * Return an instance of this class.
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor.
	 */
	private function __construct() {
		$this->includes();
		$this->init_hooks();
	}

	/**
	 * Include required files.
	 */
	private function includes() {
		require_once UDON_MP3_PLAYER_PLUGIN_DIR . 'includes/class-udon-mp3-admin.php';
		require_once UDON_MP3_PLAYER_PLUGIN_DIR . 'includes/class-udon-mp3-proxy.php';
	}

	/**
	 * Initialize hooks.
	 */
	private function init_hooks() {
		add_action( 'plugins_loaded', array( $this, 'init' ) );
		
		// 共通スクリプトの登録等
		add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
		add_action( 'enqueue_block_editor_assets', array( $this, 'enqueue_block_editor_styles' ) );
		
		// ショートコード登録
		add_shortcode( 'udon_mp3', array( $this, 'render_shortcode' ) );

		// Gutenbergブロック登録
		add_action( 'init', array( $this, 'register_block' ) );
	}

	/**
	 * Enqueue styles for Gutenberg block editor.
	 */
	public function enqueue_block_editor_styles() {
		wp_enqueue_style( 'udon-mp3-style', UDON_MP3_PLAYER_PLUGIN_URL . 'assets/css/udon-mp3-style.css', array(), UDON_MP3_PLAYER_VERSION );
	}

	/**
	 * Initialize classes.
	 */
	public function init() {
		Udon_MP3_Admin::get_instance();
		Udon_MP3_Proxy::get_instance();
	}

	/**
	 * Register Gutenberg block.
	 */
	public function register_block() {
		if ( ! function_exists( 'register_block_type' ) ) {
			return;
		}

		wp_register_script(
			'udon-mp3-block',
			UDON_MP3_PLAYER_PLUGIN_URL . 'assets/js/udon-mp3-block.js',
			array( 'wp-blocks', 'wp-element', 'wp-block-editor', 'wp-components', 'wp-server-side-render' ),
			UDON_MP3_PLAYER_VERSION,
			true
		);

		register_block_type( 'udon-mp3/player', array(
			'editor_script'   => 'udon-mp3-block',
			'render_callback' => array( $this, 'render_block' ),
			'attributes' => array(
				'audioId' => array( 'type' => 'number', 'default' => 0 ),
				'download' => array( 'type' => 'string', 'default' => 'no' ),
				'fileName' => array( 'type' => 'string', 'default' => '' ),
				'titleStr' => array( 'type' => 'string', 'default' => '' ),
			)
		) );
	}

	/**
	 * Render dynamic block.
	 */
	public function render_block( $attributes ) {
		if ( empty( $attributes['audioId'] ) ) {
			return '<p>Udon MP3 Player: 音声が選択されていません。</p>';
		}
		$atts = array(
			'id' => $attributes['audioId'],
			'download' => $attributes['download'],
			'filename' => $attributes['fileName'],
			'title' => $attributes['titleStr'],
		);
		return $this->render_shortcode( $atts );
	}

	/**
	 * Enqueue front-end scripts and styles.
	 */
	public function enqueue_scripts() {
		// Styles
		wp_enqueue_style( 'udon-mp3-style', UDON_MP3_PLAYER_PLUGIN_URL . 'assets/css/udon-mp3-style.css', array(), UDON_MP3_PLAYER_VERSION );
		
		// Scripts
		wp_enqueue_script( 'udon-mp3-script', UDON_MP3_PLAYER_PLUGIN_URL . 'assets/js/udon-mp3-script.js', array( 'jquery' ), UDON_MP3_PLAYER_VERSION, true );
		
		// 5種類のカラーセット等の設定をJSに渡す
		$color_theme = get_option( 'udon_mp3_color_theme', 'dark' );
		$player_name = get_option( 'udon_mp3_player_name', 'うどんMP3プレーヤー' );
		
		wp_localize_script( 'udon-mp3-script', 'udonMp3Settings', array(
			'ajaxurl' => admin_url( 'admin-ajax.php' ),
			'theme'   => $color_theme,
			'name'    => $player_name,
		) );
	}

	/**
	 * Render the [udon_mp3] shortcode.
	 */
	public function render_shortcode( $atts ) {
		$atts = shortcode_atts( array(
			'id'       => '',        // 添付ファイル(MP3)のID
			'download' => 'no',      // ダウンロード許可 (yes|no)
			'filename' => '',        // ダウンロード時の名前
			'title'    => '',        // 表示タイトル (空ならファイル名などから推測)
		), $atts, 'udon_mp3' );

		if ( empty( $atts['id'] ) ) {
			return '<p>Udon MP3 Player: 楽曲IDが指定されていません。</p>';
		}

		$post_id = intval( $atts['id'] );
		$file_path = get_attached_file( $post_id );

		if ( ! $file_path || ! file_exists( $file_path ) ) {
			return '<p>Udon MP3 Player: 指定されたMP3ファイルが見つかりません。</p>';
		}
		
		// MIMEタイプの確認
		$mime_type = get_post_mime_type( $post_id );
		if ( $mime_type !== 'audio/mpeg' ) {
            // 他のオーディオ形式も許容するかどうかだが、名前がMP3プレーヤーなので一旦MP3メインで。
			// return '<p>Udon MP3 Player: MP3ファイルではありません。</p>';
		}

		// メタデータの取得
		$metadata = wp_get_attachment_metadata( $post_id );
		if ( ! $metadata || ! is_array( $metadata ) ) {
		    // 代替手段として wp_read_audio_metadata() で直接読み込む
		    require_once ABSPATH . 'wp-admin/includes/media.php';
		    $metadata = wp_read_audio_metadata( $file_path );
		}

		// 必要な情報を抽出
		$sample_rate = isset( $metadata['sample_rate'] ) ? $metadata['sample_rate'] : '';
		$bitrate     = isset( $metadata['bitrate'] ) ? $metadata['bitrate'] : '';
		$channels    = isset( $metadata['channels'] ) ? $metadata['channels'] : ''; // 1=mono, 2=stereo
		$length      = isset( $metadata['length'] ) ? $metadata['length'] : 0;
		$title       = ! empty( $atts['title'] ) ? $atts['title'] : get_the_title( $post_id );

		// HQ判定 (例: 320kbps以上 または 48kHzなど)
		$is_hq = ( intval($bitrate) >= 320000 || intval($sample_rate) >= 48000 );
		$is_stereo = ( intval($channels) >= 2 );

		// プロキシURLの生成
		$download_param = ( $atts['download'] === 'yes' ) ? '1' : '0';
		$filename_param = ! empty( $atts['filename'] ) ? rawurlencode( $atts['filename'] ) : '';
		
		// nonceの生成 (セキュリティのため)
		$nonce = wp_create_nonce( 'udon_mp3_play_' . $post_id );
		
		$audio_url = admin_url( 'admin-ajax.php?action=udon_mp3_play&id=' . $post_id . '&dl=' . $download_param . '&fn=' . $filename_param . '&_wpnonce=' . $nonce );

		// プレーヤーカラー
		$color_theme = get_option( 'udon_mp3_color_theme', 'dark' );
		$player_name = get_option( 'udon_mp3_player_name', 'うどんMP3プレーヤー' );

		// HTML出力用バッファ
		ob_start();
		?>
		<div class="udon-mp3-player udon-theme-<?php echo esc_attr( $color_theme ); ?>" data-audio-url="<?php echo esc_url( $audio_url ); ?>" data-length="<?php echo esc_attr( $length ); ?>" data-download="<?php echo esc_attr( $atts['download'] ); ?>">
			<div class="udon-mp3-header">
				<span class="udon-mp3-brand"><?php echo esc_html( $player_name ); ?></span>
				<div class="udon-mp3-info-icons">
					<?php if ( $sample_rate ) : ?>
						<span class="udon-icon on"><?php echo esc_html( round( $sample_rate / 1000, 1 ) ); ?>kHz</span>
					<?php endif; ?>
					<span class="udon-icon <?php echo $is_stereo ? 'on' : 'off'; ?>"><?php echo $is_stereo ? 'STEREO' : 'MONO'; ?></span>
					<span class="udon-icon <?php echo $is_hq ? 'on' : 'off'; ?>">HQ</span>
				</div>
			</div>
			
			<div class="udon-mp3-body">
				<div class="udon-mp3-controls">
					<button class="udon-mp3-play-btn" aria-label="Play/Pause">
						<svg class="icon-play" viewBox="0 0 24 24"><path fill="currentColor" d="M8 5v14l11-7z"/></svg>
						<svg class="icon-pause" viewBox="0 0 24 24" style="display:none;"><path fill="currentColor" d="M6 19h4V5H6v14zm8-14v14h4V5h-4z"/></svg>
					</button>
				</div>
				
				<div class="udon-mp3-track-info">
					<div class="udon-mp3-title"><?php echo esc_html( $title ); ?></div>
					<div class="udon-mp3-progress-container">
						<span class="udon-mp3-time udon-mp3-current-time">0:00</span>
						<div class="udon-mp3-slider-wrap">
							<input type="range" class="udon-mp3-seek-slider" value="0" min="0" max="100" step="0.1">
							<div class="udon-mp3-buffered-bar"></div>
							<div class="udon-mp3-progress-bar"></div>
						</div>
						<span class="udon-mp3-time udon-mp3-total-time">0:00</span>
					</div>
				</div>

				<div class="udon-mp3-volume-wrap">
                    <svg class="icon-volume" viewBox="0 0 24 24"><path fill="currentColor" d="M3 9v6h4l5 5V4L7 9H3zm13.5 3c0-1.77-1.02-3.29-2.5-4.03v8.05c1.48-.73 2.5-2.25 2.5-4.02zM14 3.23v2.06c2.89.86 5 3.54 5 6.71s-2.11 5.85-5 6.71v2.06c4.01-.91 7-4.49 7-8.77s-2.99-7.86-7-8.77z"/></svg>
					<input type="range" class="udon-mp3-volume-slider" value="1" min="0" max="1" step="0.01">
				</div>

				<?php if ( $atts['download'] === 'yes' ) : ?>
				<div class="udon-mp3-download-wrap">
					<a href="<?php echo esc_url( $audio_url . '&force_dl=1' ); ?>" class="udon-mp3-download-btn" title="Download" download>
						<svg viewBox="0 0 24 24"><path fill="currentColor" d="M19 9h-4V3H9v6H5l7 7 7-7zM5 18v2h14v-2H5z"/></svg>
					</a>
				</div>
				<?php endif; ?>
			</div>
			
			<audio class="udon-mp3-audio-elem" preload="metadata"></audio>
		</div>
		<?php
		return ob_get_clean();
	}

}

// Start the plugin.
Udon_MP3_Player::get_instance();
