/**
 * Udon MP3 Player Frontend Script
 */
document.addEventListener('DOMContentLoaded', function () {

    const players = document.querySelectorAll('.udon-mp3-player');

    players.forEach(player => {
        const audio = player.querySelector('.udon-mp3-audio-elem');
        const playBtn = player.querySelector('.udon-mp3-play-btn');
        const iconPlay = playBtn.querySelector('.icon-play');
        const iconPause = playBtn.querySelector('.icon-pause');

        const seekSlider = player.querySelector('.udon-mp3-seek-slider');
        const progressBar = player.querySelector('.udon-mp3-progress-bar');
        const bufferedBar = player.querySelector('.udon-mp3-buffered-bar');

        const currentTimeEl = player.querySelector('.udon-mp3-current-time');
        const totalTimeEl = player.querySelector('.udon-mp3-total-time');

        const volumeSlider = player.querySelector('.udon-mp3-volume-slider');

        // Data attributes
        const audioUrl = player.getAttribute('data-audio-url');
        const defaultLength = parseInt(player.getAttribute('data-length'), 10) || 0;

        // State
        let isPlaying = false;
        let isSeeking = false;
        let isAudioSet = false;

        // Helper: Format Time
        function formatTime(seconds) {
            if (isNaN(seconds) || seconds < 0) return "0:00";

            const h = Math.floor(seconds / 3600);
            const m = Math.floor((seconds % 3600) / 60);
            const s = Math.floor(seconds % 60);

            if (h > 0) {
                return `${h}:${m.toString().padStart(2, '0')}:${s.toString().padStart(2, '0')}`;
            } else {
                return `${m}:${s.toString().padStart(2, '0')}`;
            }
        }

        // Initialize display
        if (defaultLength > 0) {
            totalTimeEl.textContent = formatTime(defaultLength);
        }

        // Setup audio source lazily on first play to save bandwidth
        function setupAudio() {
            if (!isAudioSet) {
                audio.src = audioUrl;
                isAudioSet = true;
            }
        }

        // Toggle Play/Pause
        playBtn.addEventListener('click', () => {
            setupAudio();

            if (isPlaying) {
                audio.pause();
            } else {
                audio.play();
            }
        });

        // Audio Events
        audio.addEventListener('play', () => {
            isPlaying = true;
            iconPlay.style.display = 'none';
            iconPause.style.display = 'block';
        });

        audio.addEventListener('pause', () => {
            isPlaying = false;
            iconPlay.style.display = 'block';
            iconPause.style.display = 'none';
        });

        audio.addEventListener('loadedmetadata', () => {
            if (audio.duration && !isNaN(audio.duration) && audio.duration !== Infinity) {
                totalTimeEl.textContent = formatTime(audio.duration);
                seekSlider.max = Math.floor(audio.duration);
            }
        });

        audio.addEventListener('timeupdate', () => {
            if (!isSeeking) {
                const current = audio.currentTime;
                currentTimeEl.textContent = formatTime(current);
                seekSlider.value = current;

                const duration = audio.duration || defaultLength || 1;
                const percent = (current / duration) * 100;
                progressBar.style.width = `${percent}%`;
            }
        });

        audio.addEventListener('progress', () => {
            if (audio.buffered.length > 0) {
                const bufferedEnd = audio.buffered.end(audio.buffered.length - 1);
                const duration = audio.duration || defaultLength || 1;
                const percent = (bufferedEnd / duration) * 100;
                bufferedBar.style.width = `${percent}%`;
            }
        });

        // Seek operations
        seekSlider.addEventListener('input', () => {
            isSeeking = true;
            currentTimeEl.textContent = formatTime(seekSlider.value);

            const duration = audio.duration || defaultLength || 1;
            const percent = (seekSlider.value / duration) * 100;
            progressBar.style.width = `${percent}%`;
        });

        seekSlider.addEventListener('change', () => {
            setupAudio();
            audio.currentTime = seekSlider.value;
            isSeeking = false;
        });

        // Volume control
        if (volumeSlider) {
            audio.volume = volumeSlider.value;
            volumeSlider.addEventListener('input', () => {
                audio.volume = volumeSlider.value;
            });
        }
    });
});
